#include "dustTagManager.h"

dustTagManager::dustTagManager()
{
	currentTagID	= 0;
	mode			= PMODE_START_MENU;
	strokeMode		= STROKE_SINGLE;
	
	screenW			= 320;
    screenH			= 480;
	
	startMenuAlpha	= 1.f;
	
	
	waitTime		= 2.f;
	waitTimer		= waitTime;
	
	bShowTimeLine	= false;
	bUploading		= false;
	bDrewUploading	= false;
	bPlayAll		= false;
	
	rotateBlend = ofxAccelerometer.getOrientation().x*2;
	
	tagPosVel.set(0,0);
	tagMoveForce = .15f;
	
	uploadStatusAlpha = 0;
	
	webResponse = "";
	bMenuOptionsOn = false;
	
	
}
//--------------------------------------------------------------
dustTagManager::~dustTagManager()
{
    
}
//--------------------------------------------------------------
void dustTagManager::setup()
{
	
	// initialize the accelerometer
	ofxAccelerometer.setup();
	ofxAccelerometer.setOrientationSmoothing(.95);
	
	// touch events will be sent to myTouchListener
	ofxMultiTouch.addListener(this);
	
	// load startup image
	bgImage.loadImage("bg2.jpg");
	
	// load fonts
	font.loadFont("frabk.ttf",28);
	fontSM.loadFont("frabk.ttf",12);
	fontSSM.loadFont("frabk.ttf",10);
	titleFont.loadFont("GARA.TTF",28);
	
	// setup gui
	gMenu.setup( ofPoint(-1,-1), 30, screenH+2);
	
	// settings for tag
	myTag.setDrawScale(screenW);
	
	// find unique id for uploading
	NSString* strId = [[UIDevice currentDevice] uniqueIdentifier];
	string uniquID = [strId UTF8String];
	gMenu.uniqueID = uniquID;
	
	// setup gml stuff
	grafXmlMaker.setup( uniquID, "Graffiti Analysis 2.0: DustTag", "1.0");

}
//--------------------------------------------------------------
void dustTagManager::reset()
{
	myTag.clear();
	
	tags.clear();
	
	currentTagID = 0;
}

static float lastTime = 0.f;

//--------------------------------------------------------------
void dustTagManager::update()
{	
	float dt  = ofGetElapsedTimef()-lastTime;
	lastTime  = ofGetElapsedTimef();
	
	bMenuOptionsOn = gMenu.isActive("OPTIONS");
	
	// check if we are ready to try uploading a tag
	if(bUploading  && bDrewUploading)
	{
		uploadTag();
		bUploading = false;
		bDrewUploading = false;
		
	}else if(!bUploading && uploadStatusAlpha > 0)
	{
		uploadStatusAlpha -= .0075;
	}
	
	
	//----- update interface elements
	if( mode == PMODE_START_MENU )
	{
		;
	}
	else if( mode == PMODE_PLAY && !bUploading)
	{
		
		// update rotation blend
		rotateBlend = .8*rotateBlend + .2*(ofxAccelerometer.getOrientation().x*3);
		
		// update pos / vel
		tags[currentTagID].position.x += tagPosVel.x;
		tags[currentTagID].position.y += tagPosVel.y;
		
		tagPosVel.x -= .1*tagPosVel.x;
		tagPosVel.y -= .1*tagPosVel.y;
				
		// update transition
		if( !myTagPlayer.bDonePlaying )									myTagPlayer.update(&tags[currentTagID]);
		if( myTagPlayer.bDonePlaying && waitTimer > 0)					waitTimer -= dt;
		else if ( myTagPlayer.bDonePlaying && lineDrawer.alpha > 0)		lineDrawer.transition(dt,.25);
		else if ( myTagPlayer.bDonePlaying )							resetPlayer();
	}
	
	
	//---------- zooming
	if( mode == PMODE_PLAY && touchOn[0] && touchOn[1] && !bMenuOptionsOn)
	{
		float dist = (touch[0].x-touch[1].x)*(touch[0].x-touch[1].x)+(touch[0].y-touch[1].y)*(touch[0].y-touch[1].y);
		dist  = sqrt(dist);
		dist /= screenH;
		
		// alter scale
		tags[currentTagID].position.z += dist-prevDist;
		tags[currentTagID].position.z  = MAX(.1,tags[currentTagID].position.z);
		
		prevDist = dist;
	}
	
	// update gui
	gMenu.update(dt);
	
}
//--------------------------------------------------------------
void dustTagManager::draw()
{
	
	ofEnableAlphaBlending();
	
	switch( mode )
	{
		case PMODE_START_MENU:
			drawStartMenu();
			break;
			
		case PMODE_REC:
			drawRecording();
			break;
			
		case PMODE_PLAY:
			drawPlayer();
			break;
	}
	
	gMenu.draw();
	
	drawUploadStatus();
	
	
}
//--------------------------------------------------------------
void dustTagManager::startRecording()
{
	mode = PMODE_REC;
	myTag.clear();
}
//--------------------------------------------------------------
void dustTagManager::touchDown(float x, float y, int touchId, ofxMultiTouchCustomData *data)
{
	if( touchId == 0 )	touchOn[0] = true;
	else				touchOn[1] = true;
	
	if( touchId == 0 )	touch[0].set(x,y);
	else				touch[1].set(x,y);
	
	
	if( touchOn[0] && touchOn[1] )
	{
		float dist = (touch[0].x-touch[1].x)*(touch[0].x-touch[1].x)+(touch[0].y-touch[1].y)*(touch[0].y-touch[1].y);
		dist  = sqrt(dist);
		dist /= screenH;
		prevDist = dist;
	}
	
	
}
//--------------------------------------------------------------
void dustTagManager::touchMoved(float x, float y, int touchId, ofxMultiTouchCustomData *data)
{
	
	if( touchId == 0 )	touchOn[0] = true;
	else				touchOn[1] = true;
	
	
	if( mode == PMODE_REC && touchId == 0 )
	{
		myTag.addNewPoint( ofPoint(x/screenW,y/screenW ),  -1, -1, -1, true);
	}
	
	if( mode == PMODE_PLAY && touchOn[0]  && touchOn[1] )
		;
	else if( touchId == 0 && mode == PMODE_PLAY && !bMenuOptionsOn)
	{
		tagPosVel.x += tagMoveForce * (x-touch[0].x);
		tagPosVel.y +=  tagMoveForce * (y-touch[0].y);
		
	}
	
	if( touchId == 0 )	touch[0].set(x,y);
	else				touch[1].set(x,y);
	
}
//--------------------------------------------------------------
void dustTagManager::touchUp(float x, float y, int touchId, ofxMultiTouchCustomData *data)
{
	
	if( touchId == 0 )	touchOn[0] = false;
	else				touchOn[1] = false;
	
	if( touchId == 0 )	touch[0].set(x,y);
	else				touch[1].set(x,y);
	
	
	
	if( strokeMode == STROKE_SINGLE && mode == PMODE_REC && myTag.myStrokes[0].pts.size() > 0 )
	{
		switchToPlayer();
		gMenu.switchButton("RECORD",false);
	}
	
	
	string menuId = gMenu.hitTest(x,y);
	
	if( menuId == "RECORD" )
	{
		if( mode == PMODE_REC ) switchToPlayer();
		else					startRecording();
		gMenu.closeOptionsMenu();
		
	}else if( mode != PMODE_REC ){
		if( menuId == "RESET" )
		{
			if( mode == PMODE_REC ) myTag.clear(true);
			else if(tags.size() > 0 ) {
				tags[currentTagID].position.set(0,0,1);
				tags[currentTagID].rotation.set(0,0,0);
			}
			
			gMenu.closeOptionsMenu();
			
		}else if( menuId == "Upload Now" && myTag.myStrokes[0].pts.size() > 0 )
		{
			bUploading = true;
			uploadStatusAlpha = 1;
			webResponse = WEB_RESPONSE_UPLOADING;
			bDrewUploading = false;
		}
		
	}
	
	gMenu.switchButton("Upload Now",false);
	
	if( gMenu.isActive("Use Multi Stroke") ) strokeMode = STROKE_MULTI;
	else strokeMode = STROKE_SINGLE;
	
	bShowTimeLine = false;
	
	if( mode == PMODE_REC )
	{
		myTag.nextStroke();
	}
	
	
	bPlayAll = gMenu.isActive( "Play All Tags");
	if(!bPlayAll && tags.size() > 0) currentTagID = tags.size()-1;
	
	if( menuId == "Clear All Tags" )
	{
		gMenu.switchButton("Clear All Tags", false);
		clearAll();
	}
	
	cout << "menuid " << menuId << endl;
}
//--------------------------------------------------------------
void dustTagManager::clearAll()
{
	if(mode == PMODE_PLAY) resetPlayer();
	
	mode			= PMODE_START_MENU;
	currentTagID	= 0;
	tags.clear();
	myTag.clear();
	
}
//--------------------------------------------------------------
void dustTagManager::touchDoubleTap(float x, float y, int touchId, ofxMultiTouchCustomData *data)
{

}
//--------------------------------------------------------------
void dustTagManager::switchToPlayer()
{
	
	// remove last stroke on tag (should be empty)
	myTag.cleanUpLastStroke();
	
	// smooth out data
	cout << "points " << myTag.myStrokes[0].pts.size() << endl;
	smoother.smoothTag(4, &myTag);
	myTag.average();
	myTag.average();
	
	// save the tag
	tags.push_back( myTag );
	
	
	// set current id to play last tag
	currentTagID = tags.size()-1;
	resetPlayer(false);
	
	// switch mode
	mode = PMODE_PLAY;

	
}
//--------------------------------------------------------------
void dustTagManager::resetPlayer( bool bUseNext )
{
	myTagPlayer.reset();
	
	// if play all
	if(bPlayAll && bUseNext) nextTag();
	
	lineDrawer.setup( &tags[currentTagID], tags[currentTagID].distMax );
	
	waitTimer = waitTime;
	
}
//--------------------------------------------------------------
void dustTagManager::nextTag()
{
	currentTagID++;
	currentTagID %= tags.size();
	
	//cout << "current "<< currentTagID << endl;
	
}
//--------------------------------------------------------------
void dustTagManager::drawStartMenu()
{

	glDisable(GL_DEPTH_TEST);
	
	glColor4f(1,1,1,1);
	bgImage.draw(0,0,320,480);
	
}

//--------------------------------------------------------------
void dustTagManager::drawRecording()
{
	ofSetColor(255,255,255,255);
	myTag.draw();
	
	
	if( myTag.myStrokes[0].pts.size()>0)
	{
		
		float time = ofGetElapsedTimef() - myTag.startTime;
		drawTime( time );
	}

}
//--------------------------------------------------------------
void dustTagManager::drawPlayer()
{
	ofSetColor(255,255,255,255);
	
	glPushMatrix();
	
	glTranslatef(screenW/2, screenH/2, 0);
	
	glScalef(tags[currentTagID].position.z,tags[currentTagID].position.z,tags[currentTagID].position.z);
	
	glTranslatef(tags[currentTagID].position.x,tags[currentTagID].position.y,0);
		
	glPushMatrix();
		
		glRotatef(tags[currentTagID].rotation.y+(rotateBlend)+10,1,0,0);
		glTranslatef(-tags[currentTagID].min.x*tags[currentTagID].drawScale,-tags[currentTagID].min.y*tags[currentTagID].drawScale,-tags[currentTagID].min.z);
		glTranslatef(-tags[currentTagID].center.x*tags[currentTagID].drawScale,-tags[currentTagID].center.y*tags[currentTagID].drawScale,-tags[currentTagID].center.z);

		//tags[currentTagID].draw();
	
		glPushMatrix();
			glScalef( tags[currentTagID].drawScale, tags[currentTagID].drawScale, 1);
			lineDrawer.draw( myTagPlayer.getCurrentStroke(), myTagPlayer.getCurrentId() );
		glPopMatrix();
	glPopMatrix();
	
	glPopMatrix();
	
	
	// draw time
	if( !bMenuOptionsOn) drawTime( myTagPlayer.getCurrentTime() );
	
	
}

void dustTagManager::drawTime(float time)
{
	float wd = font.stringWidth( ofToString( time,2) ) / 10.f;
	wd = 10*(int)(wd);
	
	// draw time
	glPushMatrix();
	
	glTranslatef(screenW/2.f, screenH/2.f,0);
	glRotatef(90, 0, 0, 1);
	glTranslatef(-screenH/2.f, -screenW/2.f,0);
	
	ofSetColor(255,255,255,255);
	glTranslatef(screenH-wd,screenW-10,0);
	glScalef(.8,.8,1);
	
	
	
	font.drawString( ofToString( time,2), 0,0);
	
	glPopMatrix();
}

void dustTagManager::drawUploadStatus()
{
	
		glPushMatrix();
		
		glTranslatef(screenW/2.f, screenH/2.f,0);
		glRotatef(90, 0, 0, 1);
		glTranslatef(-screenH/2.f, -screenW/2.f,0);
		
		
		if( bUploading || uploadStatusAlpha > 0)
		{
			ofSetColor(255,255,255,255*uploadStatusAlpha);
			fontSM.drawString( webResponse, 270, 194 );//screenH/2-50,screenW/2-30);
			bDrewUploading = true;
		}
	
	
		glPopMatrix();
		
		
		
	
}

void dustTagManager::uploadTag()
{
	
	
	webResponse.clear();
	
	// first make a quick test
	vector<string> keys;
	vector<string>vals;
	
	// prepare and upload tag if all is good
	grafXmlMaker.clearKeywords();
	grafXmlMaker.addKeyword(gMenu.keywords);
		
	string gmlData;
	grafXmlMaker.constructGML(&tags[currentTagID]);
	grafXmlMaker.xml.copyXmlToString(gmlData);
				
	keys.push_back("gml");
	vals.push_back(gmlData);
		
	string response = wUploader.send(UPLOAD_URL, keys,vals,WEB_POST);
	
	// note: edit this to receive response fromserver
	int num = response.find_first_of("response");
	if( num<=0 ) webResponse = WEB_RESPONSE_SUCCESS;
	else webResponse = WEB_RESPONSE_FAIL;
		
	
	bUploading = false;
	
	
}
