#include "grafManager.h"

bool sort_zpos_compare( timePt a, timePt b ) {
	
	return (a.pos.z < b.pos.z);
}


grafManager::grafManager()
{
	recorder		 = new grafRecorder();
	vizStroke		 = new grafVStroke();
	vizParticleField = new grafVParticleField();
	io				 = new grafIO();
	pocoLister		 = new ofxPocoDirectoryLister();
	
	screenW			= 1024;
    screenH			= 768;
	recScale		= 1024;
		
	bPlaying		= true;
	bReadyToRecord	= false;
	
	currentTag		= 0;
	rotation		= 0.f;
	
	waitTime    = 2.0f;
    waitCounter = waitTime;
	lastTime    = ofGetElapsedTimef();
	dt = 0.f;
	
	avgRate		= .25;
	fadeRate	= 8.f;
	
	mode	= MODE_NORM;
	bPanelOn = false;
	
	franklinBook.loadFont("frabk.ttf",28);
	franklinBookSM.loadFont("frabk.ttf",14);
	franklinBookSSM.loadFont("frabk.ttf",10);
	
	nRecordings = 0;
	
}
//--------------------------------------------------------------
grafManager::~grafManager()
{
    delete recorder;
	delete vizStroke;
	delete vizParticleField;
	delete io;
	//delete gui;
	delete pocoLister;
}
//--------------------------------------------------------------
void grafManager::setup()
{
	
	// set recorder sacle for this session (may change from screen size)
	recorder->tag.setDrawScale(recScale);
	recorder->scale = recScale;
	
	// set up particles
	vizParticleField->setup(screenW,screenH);
	
	// set up gui
	setupGuiControls();
	
		
}
//--------------------------------------------------------------
void grafManager::reset()
{
	myTags.clear();

}
//--------------------------------------------------------------
void grafManager::setupGuiControls()
{
	panel.setup("grafAnalysis", 10, 10, 300, 600);
	//panel.addPanel("Playback", 1, false);
	panel.addPanel("Display", 1, false);
	
	//---- playback panel
	/*panel.setWhichPanel("Playback");
	panel.setWhichColumn(0);
	panel.addToggle("Prev Tag","PREV_TAG",false);
	panel.addToggle("Next Tag","NEXT_TAG",false);
	panel.addToggle("Save Tag","SAVE_TAG",false);*/
	
	
	panel.setWhichPanel("Display");
	panel.setWhichColumn(0);
	panel.addSlider("Particle Damping","P_DAMPING",.18f,0,1,false);
	panel.addSlider("Particle Size","P_SIZE",3,0,20,false);
	
}
//--------------------------------------------------------------
void grafManager::update()
{
	
	dt  = ofGetElapsedTimef()-lastTime;
	lastTime  = ofGetElapsedTimef();
	
	//---------- playback
	if( mode == MODE_TRANS || mode == MODE_NORM )
	{
			
		
		if( bPlaying && myTags.size() > 0 && !myTags[ currentTag ].bPaused )
		{
			bool bUpdateParticles = false;
		
			// check current playback state
			// if tag finished drawing, pause
			if( myTags[ currentTag ].bReset )
			{
			
				if( waitCounter <= 0 )  
				{	
					mode = MODE_TRANS; // if pause is done, start transition mode
				
				}else{
					
					// if waiting, decrease wait time, cont update particle system, remove arrow
					waitCounter = (waitCounter-dt < 0) ? 0 : waitCounter-dt;
					bUpdateParticles = true;
				}
			
			}else{
           
				myTags[ currentTag ].update();
				if(myTags[ currentTag ].getCurrentId() > 1 && mode != MODE_TRANS) 
					bUpdateParticles = true;
			}
		
			
			if( mode == MODE_TRANS && updateTransitions( &myTags[currentTag],dt) )
			{
				bUpdateParticles = true;
				nextTag();
			}
		
			//if(bUpdateParticles)
				vizParticleField->update(&myTags[ currentTag ],myTags[ currentTag ].getCurrentPoint(),myTags[ currentTag ].getVelocityForTime( myTags[ currentTag ].getCurrentTime() ), dt);
		
		
		}
	
		
	}
	
	// calculate timing on rotation and update
	updateRotation(&myTags[ currentTag ]);
	
	cout << " current point " << myTags[currentTag].getCurrentId() << endl; 

	
	
	updateGui();
	

}

static float rvel = 0.f;
static float counterR = 0.f;

void grafManager::updateRotation(grafTag * tag)
{
    float totalTime = tag->getDuration() + waitTime;
    float nowTime   = (waitTime-waitCounter) + tag->getCurrentTime();
    float pctDone   = nowTime/totalTime;
	
    if( mode == MODE_NORM )
    {
        rotation = -25 * powf(1-(pctDone),1.15);
        rvel = rotation;
        counterR = 0;
    }else{
		counterR += .5 * dt;
		rotation = 25 * powf(counterR,1.15) + rvel;
		
    }
	
	//cout << "rotation " << rotation << " dt " << dt << " " << rvel << endl;
	
	if(rotation < -360) rotation = -360;
	if(rotation > 360 ) rotation = 360;
	//rotation = MIN(rotation,360);
	//rotation = MAX(rotation,0);
}

//--------------------------------------------------------------
void grafManager::updateGui()
{
	//--- update gui
	panel.update();
	
	vizParticleField->setDamping(panel.getValueF("P_DAMPING"));
	vizParticleField->setParticleSize(panel.getValueF("P_SIZE"));
	
	// check panels
	/*if( panel.getValueB("NEXT_TAG") )
	{
		nextTag();
		panel.setValueB("NEXT_TAG",false);
	}
	
	else if( panel.getValueB("PREV_TAG") )
	{
		prevTag();
		panel.setValueB("PREV_TAG",false);
	}
	
	if( panel.getValueB("SAVE_TAG") )
	{
	   io->saveTagPreservePts( &myTags[currentTag], "tags/"+myTags[currentTag].tagname);
	   panel.setValueB("SAVE_TAG",false);
	
	}*/
	
}
//--------------------------------------------------------------
void grafManager::draw()
{
	
	
	ofEnableAlphaBlending();
	
	if(mode == MODE_REC)
		
		drawRecording();
	
	else /*if ( mode == MODE_TRANS || mode == MODE_NORM )*/
		
		drawPlayback();
	
	
	if(bPanelOn) panel.draw();
	

}
//--------------------------------------------------------------
void grafManager::drawPlayback()
{
	
	
	//cout << " draw playback " << endl;
	
	glPushMatrix();
		
		
	
		//---------- set up draw pos, rot, scale
		// center screen
		glTranslatef(screenW/2, screenH/2, 0);
		
		//glMultMatrixf(ofxAccelerometer.getMatrix());	
		glScalef(myTags[currentTag].position.z,myTags[currentTag].position.z,myTags[currentTag].position.z);
			
		// rotate
		glRotatef(myTags[currentTag].rotation.x+rotation,0,1,0);
		glRotatef(myTags[currentTag].rotation.y,1,0,0);
		glRotatef(myTags[currentTag].rotation.z,0,0,1); 
			
		// translate to draw offset
		glTranslatef(myTags[currentTag].position.x,myTags[currentTag].position.y,0);
	
		
		vizParticleField->draw(&myTags[ currentTag ],screenW,screenH);
	
		//---------- playback
		if( bPlaying )
		{		
			vizStroke->draw(&myTags[ currentTag ],-screenW/2, -screenH/2);
		}
	
	glPopMatrix();
	
	// draw gui
	ofSetColor(0xffffff);
	//gui->draw();
	
	
	

}
//--------------------------------------------------------------
void grafManager::drawRecording()
{
	// draw recording line
	glPushMatrix();
	
		glTranslatef(screenW/2, screenH/2, 0);
		recorder->draw(-screenW/2, -screenH/2);
	
	glPopMatrix();
	
	
}
//--------------------------------------------------------------
bool grafManager::updateTransitions( grafTag * tag, float dt )
{
	
	float ddt = dt*(1/16.f);
	bool bDoneWithTag = false;
	
	vizStroke->averagePoints(avgRate);
	vizStroke->alterParam( -(.75*fadeRate) * ddt, vizStroke->alphaStroke);
	
	if( vizStroke->alphaStroke < 1 ){
		vizStroke->alterParam( -(.75*fadeRate) * ddt, vizStroke->alpha);
		if( vizStroke->alpha <= 0 ) bDoneWithTag =  true;
		//vizParticleField->retractParticles( ddt, screenW*((tag->max.x-tag->min.x) / 2),screenW*((tag->max.y-tag->min.y) / 2) );

	}
	
	cout << vizStroke->alpha <<  " transition " << bDoneWithTag << endl;
	return bDoneWithTag;
}
//--------------------------------------------------------------
void grafManager::storeCurrentRecording()
{
	if( recorder->tag.pts.size() == 0 ) return;
	
	grafTag tempTag;
	
	myTags.push_back( tempTag );
	myTags[ myTags.size()-1 ] = recorder->tag;
	myTags[ myTags.size()-1 ].setDrawScale( recScale );
	//myTags[ myTags.size()-1 ].averagePts();
	
	currentTag = myTags.size()-1;
	
	recorder->smooth(&myTags[currentTag]);
	
	//myTags[ currentTag ].tagname = io->getTagName( myTags[ currentTag ].tagname );
	
	nRecordings++;
	//if(nRecordings==1) gui->setDoubleButton("Record",1);
	//io->saveTag(&myTags[currentTag]);
}
//--------------------------------------------------------------
void grafManager::adjustTagPos( int x, int y, int z )
{
	if(myTags.size() < currentTag || myTags.size() == 0 ) return;
	
	myTags[currentTag].position.x += x;
	myTags[currentTag].position.y += y;
	myTags[currentTag].position.z += z*.01f;
	
	//myTags[currentTag].position.x = MIN( MAX(-160,myTags[currentTag].position.x), 160);
	//myTags[currentTag].position.y = MIN( MAX(-220,myTags[currentTag].position.y), 220);
}
//--------------------------------------------------------------
void grafManager::adjustTagRot( int x, int y )
{
    if(myTags.size() < currentTag ) return;
	
	myTags[currentTag].rotation.x += x;
	myTags[currentTag].rotation.y += y;
	
}
//--------------------------------------------------------------
void grafManager::resetTag()
{
	// reset tag to start point
    if(myTags.size() == 0 ) return;	
	
	myTags[currentTag].resetToStart();

}
//--------------------------------------------------------------
void grafManager::nextTag()
{
	resetTag();
	
	myTags[currentTag].bReset    = false;
    waitCounter                  = waitTime;
	
    //currentTag++;
    //currentTag = currentTag % (int)myTags.size();
	
    //resetTag();
	vizParticleField->bResetPS = true;
	vizParticleField->reset();
    vizStroke->reset();
	
    mode = MODE_NORM;
}
//--------------------------------------------------------------
void grafManager::prevTag()
{
	resetTag();
	
	myTags[currentTag].bReset    = false;
    waitCounter                  = waitTime;
	
    currentTag--;
    if( currentTag < 0 ) currentTag = myTags.size()-1;
	
    resetTag();
	
	vizParticleField->reset();
    vizStroke->reset();
	
    mode = MODE_NORM;
}
//--------------------------------------------------------------
void grafManager::loadTag( string filename )
{
	grafTag tempRec;
	
	myTags.push_back( tempRec );
	int i = myTags.size()-1;
	
	//io->loadSavedTag( &myTags[i], filename+".gml" );
	io->loadSavedTag( &tempRec, filename+".gml" );
	myTags[i] = tempRec;
	recorder->smoother.smoothTag( 4, &tempRec, &myTags[i] );
	sort( myTags[i].pts.begin(), myTags[i].pts.end(), sort_zpos_compare );
	myTags[i].calcMinMax();
	myTags[i].averagePts();
	myTags[i].averagePts();
	
	//recorder->smooth(&myTags[i]);
	//myTags[i].setCurrentToLast();
	myTags[i].drawScale = 1024;
	myTags[i].tagname = filename;
	
	resetTag();
	currentTag = i;
	resetTag();
	
	vizParticleField->bResetPS = true;
	vizParticleField->reset();
    vizStroke->reset();
	
    mode = MODE_NORM;
}
//--------------------------------------------------------------
void grafManager::loadAllSavedTags( string directory, bool bUseFolders )
{
	// load all tags in the give directory
	dirs.clear();
	
	if( bUseFolders )
	{
		// find all folders in directory "tags"
		pocoLister->setPath( ofToDataPath(directory+"/",true) );
		pocoLister->findSubDirectories(dirs);
		
		// check that the folders have tag file
		for( int i = dirs.size()-1; i >= 0; i--)
		{
			pocoLister->setPath( ofToDataPath(directory+"/"+dirs[i]+"/",true) );
			pocoLister->setExtensionToLookFor("gml");
			int num = pocoLister->getNumberOfFiles();
			if( num <= 0 ) dirs.erase( dirs.begin()+i);
		}
		
	}else{
		pocoLister->setPath( ofToDataPath(directory+"/",true) );
		pocoLister->setExtensionToLookFor("gml");
		pocoLister->findFiles(dirs);
	}
	
	
	
    if( dirs.size() <= 0 ) return;
	
    // sort alphabetically
    std::sort(dirs.begin(),dirs.end());
	
    // clear and load each tag
    reset();
	
	
	grafTag tempRec;
	
    for( int i = 0; i < dirs.size(); i++)
    {
		
        // set up smoothed tags
        myTags.push_back( tempRec );
		
		string filename;
		if( bUseFolders ) filename =  directory + "/"+dirs[i] + "/" + dirs[i]+".gml";
		else filename =  directory + "/"+dirs[i];
		
		io->loadSavedTag( &myTags[i], filename );
        
		recorder->smooth(&myTags[i]);
		myTags[i].setCurrentToLast();
		myTags[i].drawScale = 1024;
		myTags[i].tagname = dirs[i];
		
	}
}
